
#include <config.h>

#define BPF_MAJOR_VERSION
#include <pcap.h>


//#include <windows.h>
//#include <tchar.h>


#include "err.h"
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <malloc.h>

#include <syslog.h>


#pragma warning(disable: 4200)

/* XXX - libevent */
#undef timeout_pending
#undef timeout_initialized

extern "C" {
#include <event.h>
#include <dnet.h>

extern "C" extern int event_gotsig;
extern "C" extern int (*event_sigcb)(void);

}

#include <sys/tree.h>

#include "arpd.h"
static int			 arpd_sig;



#include "cmdline.h"
#include "eventlog.h"
#include "scm.h"
#include "arpdconfig.h" 
#include "service.h"


int print_usage();
int print_service_status();
void update_config(CommandLine& cmdLine, ArpdConfig& cfg);
void print_interfaces();
void WINAPI service_main(DWORD argc, LPTSTR *argv);
void WINAPI service_control_handler(DWORD opcode);
unsigned int __stdcall arpd_run( ArpdConfig* cfg );
unsigned int __stdcall arpd_run_proxy(ArpdConfig* cfg );
int __stdcall service_status_callback(SC_HANDLE sch, SERVICE_STATUS *, long timeValue);


# define SERVICE_RUN			_T("start-service")
# define REGISTRY_CONFIG_KEY	_T("software\\netvigilance\\winarpd")


class ExceptionNoAddresses
{
};


class ExceptionServiceStartFailed
{
public:
	ExceptionServiceStartFailed(LONG _result) : result(_result)
	{}

public:
	LONG result;
};

class ExceptionServiceStopFailed
{
public:
	ExceptionServiceStopFailed(LONG _result) : result(_result)
	{}

public:
	LONG result;
};

class ExceptionServiceUnisntallFailed
{
public:
	ExceptionServiceUnisntallFailed(LONG _result) : result(_result)
	{}

public:
	LONG result;
};

class ExceptionServiceIsntallFailed
{
public:
	ExceptionServiceIsntallFailed(LONG _result) : result(_result)
	{}

public:
	LONG result;
};



SERVICE_TABLE_ENTRY service_dispatch_table[] = 
{
	{_T("winarpd"), service_main },
	{0, 0 }
};

int _tmain(DWORD argc, TCHAR *argv[])
{
	CmdOption options[] = 
	{
		{"h",		  "h", no_argument},
		{"?",		  "h", no_argument},
		{"help",	  "h", no_argument},
		{"d",			0, no_argument},
		{"W",			0, no_argument},
		{"I",			0, argument_optional},
		{"S",			0, no_argument},
		{"T",			0, no_argument},
		{"U",			0, no_argument},
		{"R",			0, no_argument},

		{"r",			0, argument_required},
		{"i",			0, argument_required},
		{"l",			0, argument_required},
		{SERVICE_RUN,	0, no_argument},

		{0}
	};

	CommandLine cmdLine( options );

	try
	{
	cmdLine.Parse(argc, argv);

	if (cmdLine.Empty())
		return print_service_status();

	if (cmdLine.Conains(_T("h")))
		return print_usage();

	if (cmdLine.Conains(SERVICE_RUN))
	{
		if (!::StartServiceCtrlDispatcher(service_dispatch_table))
		{
			TCHAR buffer[1024];
			snprintf(buffer, sizeof(buffer)-1, _T("Faield to start service: %x\n"), GetLastError());
			::OutputDebugString(buffer);
			//errx(1, buffer);
		}
		return 0;
	}

	if (cmdLine.Conains(_T("W")))
	{
		print_interfaces();	
		return 0;
	}

	if (cmdLine.Conains(_T("T")) || cmdLine.Conains(_T("U")) ) /* call SCM to stop service */
	{	
		printf("Stopping %s service.", service_dispatch_table[0].lpServiceName);

		//SCM::StopService( service_dispatch_table[0].lpServiceName, 1000, service_status_callback );
		if (!SCM::StopService( service_dispatch_table[0].lpServiceName, 1000, service_status_callback ))
			throw ExceptionServiceStopFailed( GetLastError() );
		printf("done.\n");

		if (cmdLine.Conains(_T("T")))
			return 0;

		/* uninstall the service 
		 * */
		printf("Uninstalling %s service.", service_dispatch_table[0].lpServiceName);

		EventLog::DeregisterSource( service_dispatch_table[0].lpServiceName );
		if (!SCM::UninstallService( service_dispatch_table[0].lpServiceName ))
			throw ExceptionServiceUnisntallFailed( GetLastError() );
		printf("done.\n");

		return 0;
	}


	/*
	 * because of -I (must), -S & -R options provides new parameters to service 
	 * */
	ArpdConfig cfg;
	if (cmdLine.Conains(_T("S")) || cmdLine.Conains(_T("R")))
		ArpdConfig::Load(REGISTRY_CONFIG_KEY, cfg);

	/* parse command line arguments into config 
	 * */
	update_config(cmdLine, cfg);


	/* make sure that at least 1 ip addrss is specified into cmd line
	 * */
	if (cmdLine.Conains(_T("I")))
	{
		if (cfg.Addresses().Count() == 0)
			throw ExceptionNoAddresses();
	}


	if (cmdLine.Conains(_T("d")))
	{
		debug = 1;

		openlog(cfg.GetLogPath(), 0, 0);

		return arpd_run( &cfg );
	}


	ArpdConfig::Save(REGISTRY_CONFIG_KEY, cfg);


	if (cmdLine.Conains(_T("I")))
	{
		/* 1. parse config must be present
		 * 2. save config to registry
		 * 3. register service into SCM
		 * */

		/* determine startup mode
		 * */
		int startupMode = SERVICE_AUTO_START;
		if (cmdLine[_T("I")].Value() != 0)
			startupMode = tolower(cmdLine[_T("I")].Value()[0]) == _T('m') ? SERVICE_DEMAND_START : SERVICE_AUTO_START; 

		/*
		 * create service command line 
		 * */
		TCHAR svcCmdLiime[10240];
		::GetModuleFileName(NULL, svcCmdLiime, sizeof(svcCmdLiime)/sizeof(TCHAR));
		strcat(svcCmdLiime, " -"SERVICE_RUN);
		
		printf("Installing %s service.", service_dispatch_table[0].lpServiceName);

		if (!SCM::InstallService(
				svcCmdLiime, 
				service_dispatch_table[0].lpServiceName, 
				service_dispatch_table[0].lpServiceName, startupMode))
			throw ExceptionServiceIsntallFailed(GetLastError());
		printf("done.\n");

		EventLog::RegisterSource(argv[0], service_dispatch_table[0].lpServiceName, 7);
		if (cmdLine.Conains(_T("S")))
		{
			printf("Starting %s service.", service_dispatch_table[0].lpServiceName);
			if (!SCM::StartService( service_dispatch_table[0].lpServiceName, 1000, service_status_callback  ))
				throw ExceptionServiceStartFailed(GetLastError());
			printf("done.\n");
		}
		return 0;
	}

	if (cmdLine.Conains(_T("S")))
	{
		/* 1. parse config (optional)
		 * 2. save config to registry (if config supplied)
		 * 3. call SCM to start service
		 * */
		if (cfg.Addresses().Count() == 0)
			throw ExceptionNoAddresses();

		printf("Starting %s service.", service_dispatch_table[0].lpServiceName);
		if (!SCM::StartService( service_dispatch_table[0].lpServiceName, 1000, service_status_callback ))
			throw ExceptionServiceStartFailed(GetLastError());
		printf("done.\n");

		return 0;
	}

	if (cmdLine.Conains(_T("R")))
	{
		/* 1. parse config (optional)
		 * 2. save config to registry (if config supplied)
		 * 3. call SCM to stop service
		 * 4. call SCM to start service
		 * */
		if (cfg.Addresses().Count() == 0)
			throw ExceptionNoAddresses();

		printf("Stopping %s service.", service_dispatch_table[0].lpServiceName);
		if (!SCM::StopService( service_dispatch_table[0].lpServiceName, 1000, service_status_callback ))
			throw ExceptionServiceStopFailed( GetLastError() );
		printf("done.\n");

		printf("Starting %s service.", service_dispatch_table[0].lpServiceName);
		if (!SCM::StartService( service_dispatch_table[0].lpServiceName, 1000, service_status_callback  ))
			throw ExceptionServiceStartFailed(GetLastError());
		printf("done.\n");

		return 0;
	}
		

	
	}
	catch( ExceptionIndexOutOfBounds )
	{
	}
	catch( ExceptionOptionNotFound e )
	{
		printf("options '%s' doesn't exist into command line\n", e.Name());
	}
	catch(ExceptionUnknownOption e) 
	{
		printf( "comand line contains unknown option %s\n", e.Name());
	}
	catch(ExceptionMandatoryArguemntMissing e) 
	{
		printf( "option %s - missing required arguent\n", e.Name());
	}
	catch( ExceptionNoAddresses )
	{
		printf( "There is no addresses into command line\n");
	}
	catch( ExceptionSaveConfig )
	{
		printf( "Failed to save config to registry\n");
	}
	catch( ExceptionLoadConfig )
	{
		printf( "Failed to load config from registry\n");
	}
	catch( ExceptionSCMOpenFailed e)
	{
		printf( "Connect to SCM failed %d\n", e.result);
	}
	catch( ExceptionSCMOpenServiceFailed e)
	{
		printf( "Failed open service %d\n", e.result);
	}
	catch( ExceptionSCMQueryServiceFailed e)
	{
		printf( "Failed to query service status %d\n", e.result);
	}
	catch( ExceptionSCMControlServiceFailed e)
	{
		printf( "Failed to send control request to service %d\n", e.result);
	}
	catch( ExceptionServiceUnisntallFailed e)
	{
		printf( "Failed to unistall service %d\n", e.result);
	}
	catch( ExceptionServiceIsntallFailed e)
	{
		printf( "Failed to istall service %d\n", e.result);
	}
	catch( ExceptionServiceStopFailed e)
	{
		printf( "Failed to stop service %d\n", e.result);
	}
	catch(ExceptionServiceStartFailed e)
	{
		printf( "Failed to start service %d\n", e.result);
	}
	
	return 0;
}

int __stdcall service_status_callback(SC_HANDLE sch, SERVICE_STATUS *, long timeValue)
{
	putchar('.');
	return timeValue <= 30000;
}

/* extracts configuration argimrns from command line
 * installs information from commad line and updates 
 * config
 * */
void update_config(CommandLine& cmdLine, ArpdConfig& cfg)
{
	if (cmdLine.Conains(_T("i")))
	{
		cfg.Devices().Clear();

		Arguments a;
		cmdLine.Enumerate(_T("i"), a);
		for(int i = 0; i < a.Count(); ++i)
			cfg.Devices().Add(a[i].Value());

	}

	if (cmdLine.Conains(_T("l")))
		cfg.LogFile() = cmdLine[_T("l")].Value();

	if (cmdLine.Conains(_T("r")))
		cfg.MaxRequestNumber() = atoi( cmdLine[_T("r")].Value() );

	if ( cmdLine.Unnamed().Count() != 0)
		cfg.Addresses().Clear();

	for(int i = 0; i < cmdLine.Unnamed().Count(); ++i ) 
	{
		cfg.Addresses().Add( cmdLine.Unnamed()[i].Value() ); 
	}


}

int print_usage()
{

	printf(
//"Usage: arpd -dWSTUR [-l logFile] [-I [ startup ]] -i interface [-i interface]  net [net]\n"
"Usage: winarpd -dWSTUR [-l logFile] [-I [ startup ]] [-i interface] net [net]\n"
"Where:\n"
"   -W - Displays all available network interfaces\n"
"   -I [startup] - Installs winarpd as win32 service process. The startup\n"
"      parameter specified startup type and can be either automatic\n"
"      or manual\n"
"   -S - Starts winarpd system service\n"
"   -T - Stops winarpd service process\n"
"   -R - Restarts (stop and then start) winarpd service process\n"
"   -U - Uninstall winarpd service process\n"
"   -d - run winarpd as standalone app\n"
"   -l - Log file path\n"
"   -i interface - Specifies network interface to bind arpd to\n"
"   -r number - Specifies number of arp requests to pass before report\n"
"      self address. Default is 3.\n"
"    net - specifies list of ip addresses to respond\n"
"\n"
);

	return 1;
}

int print_service_status()
{
	printf(
	"WinArpd win32 service V 1.1b5\n"
	"Original arpd\n"
	"Copyright (c) 2001, 2002 Dug Song <dugsong@monkey.org>\n"
	"Copyright (c) 2002 Niels Provos <provos@citi.umich.edu>\n"
	"WinArpd\n"
	"Copyright (c) 2005 Jesper Jurcenoks & Yurii Ingultsov\n"
	"\n"
	);


	try
	{
		SERVICE_STATUS ss;
		TCHAR *serviceStatus = _T("unknown");
		switch(SCM::QueryStatus(service_dispatch_table[0].lpServiceName, &ss))
		{
		case SERVICE_STOPPED:			serviceStatus = _T("stopped");		break;
		case SERVICE_START_PENDING:		serviceStatus = _T("starting");		break;
		case SERVICE_STOP_PENDING:		serviceStatus = _T("stoping");		break;
		case SERVICE_RUNNING:			serviceStatus = _T("running");		break;
		case SERVICE_CONTINUE_PENDING:	serviceStatus = _T("resuming");		break;
		case SERVICE_PAUSE_PENDING:		serviceStatus = _T("suspending");	break;
		case SERVICE_PAUSED:			serviceStatus = _T("paused");		break;
		}

		_tprintf("winarpd service................. %s.\n", serviceStatus );

		ArpdConfig cfg;
		ArpdConfig::Load(REGISTRY_CONFIG_KEY, cfg);

		_tprintf(_T("winarpd log file................ %s.\n"), cfg.GetLogPath());
		_tprintf(_T("winarpd listening on interface.. %s.\n"), cfg.Devices().Count() ? cfg.Devices()[0]: _T("not specified"));
		_tprintf(_T("winarpd respond on every ....... %dth request.\n"), cfg.MaxRequestNumber());
		_tprintf(_T("winarpd responding on........... %s.\n"), cfg.Addresses().Count() ? cfg.Addresses()[0]: _T(""));
		for(int i = 1; i < cfg.Addresses().Count(); ++i)
		_tprintf(_T("                              %s.\n"), cfg.Addresses()[i]);

	}
	catch( ExceptionSCMOpenFailed e)
	{
		printf("Failed to connect to SCM %d\n", e.result);
	}
	catch( ExceptionSCMOpenServiceFailed)
	{
		printf("winarpd service not installed\n");
	}
	catch(ExceptionSCMQueryServiceFailed e)
	{
		printf("Failed to query service status %d\n", e.result);
	}
	catch(ExceptionLoadConfig)
	{
		printf("failed to load arpd config");
	}

	printf("\n");

	return 1;
}


void print_interfaces()
{
	pcap_if_t *alldevs, *d;
	char ebuf[10240];
	int i;

	if (pcap_findalldevs(&alldevs, ebuf) == -1)
		errx(1, "pcap_findalldevs: %s", ebuf);

	printf("\nInterface\tDevice\t\tDescription\n-------------------------------------------\n");
	for(i = 1, d=alldevs;d;d=d->next, i++) 
	{
		printf("%d %s",i, d->name);

		if (d->description)
			printf("\t%s",d->description);

		printf("\n");
	}
}

/* 
 * called from service control manager in response to call  
 * StartServiceControlDispatcher() in main function
 *
 * */
ServiceStatus serviceStatus;
void WINAPI service_main(DWORD argc, LPTSTR *argv)
{

	serviceStatus.RegisterServiceCtrlHandler(service_dispatch_table[0].lpServiceName, service_control_handler);
	serviceStatus.ReportStatus(SERVICE_RUNNING);

	try
	{

		ArpdConfig cfg;
		ArpdConfig::Load(REGISTRY_CONFIG_KEY, cfg);

		openlog(cfg.GetLogPath(), 0, 0);
		syslog(LOG_INFO, "arpd service started");

		if (cfg.Addresses().Count() == 0)
			throw ExceptionNoAddresses();


		arpd_run_proxy(&cfg);
	
	}
	catch(ExceptionLoadConfig)
	{
		errx(1, "Failed to load config");
	}
	catch(ExceptionNoAddresses)
	{
		errx(1, "Invalid configuration\n");
	}

	serviceStatus.ReportStatus(SERVICE_STOPPED);

}




void WINAPI service_control_handler(DWORD fdwControl)
{
	switch(fdwControl)
	{
	case SERVICE_CONTROL_STOP:
	case SERVICE_CONTROL_SHUTDOWN: 
		//SetEvent( g_hTerminateEvent ) ;
		raise(SIGINT);
		break;
	default:
		break;
	}
}


void _terminate_handler(int sig)
{

	event_gotsig = 1;
	arpd_sig = sig;
}


int __cdecl _signal_handler(void)
{
	syslog(LOG_INFO, "exiting on signal %d", arpd_sig);
	serviceStatus.ReportStatus(SERVICE_STOPPED);
	arpd_exit(0);
	/* NOTREACHED */
	return (-1);
}

const char *ExceptionCodeToString(DWORD dwCode)
{
switch(dwCode)
{
case EXCEPTION_ACCESS_VIOLATION:			return "Access Violation";
case EXCEPTION_ARRAY_BOUNDS_EXCEEDED:		return "Array Out of Bounds";
case EXCEPTION_BREAKPOINT:					return "Breakpoint";
case EXCEPTION_DATATYPE_MISALIGNMENT:		return "Datatype Managment";
case EXCEPTION_FLT_DENORMAL_OPERAND:		return "Floating Point Denormal Operand";
case EXCEPTION_FLT_DIVIDE_BY_ZERO:			return "Floating Point Divide by Zero";
case EXCEPTION_FLT_INEXACT_RESULT:			return "Floating Point Inexact result";
case EXCEPTION_FLT_INVALID_OPERATION:		return "Floafing Point Invalid Operation";
case EXCEPTION_FLT_OVERFLOW:				return "Floating Point Overflow";
case EXCEPTION_FLT_STACK_CHECK:				return "Floating Point Stack Check";
case EXCEPTION_FLT_UNDERFLOW:				return "Floafing Point Underflow";
case EXCEPTION_ILLEGAL_INSTRUCTION:			return "Illegal Instructuion";
case EXCEPTION_IN_PAGE_ERROR:				return "In page error";
case EXCEPTION_INT_DIVIDE_BY_ZERO:			return "Integer Divide by Zero";
case EXCEPTION_INT_OVERFLOW:				return "Integer Overflow";
case EXCEPTION_INVALID_DISPOSITION:			return "Invalid Disposition";
case EXCEPTION_NONCONTINUABLE_EXCEPTION:	return "Non continuable exception";
case EXCEPTION_PRIV_INSTRUCTION:			return "Proviledged Instruction";
case EXCEPTION_SINGLE_STEP:					return "Single Step";
case EXCEPTION_STACK_OVERFLOW:				return "Stack Overflow";
default:
	return "Unknown Exception";
}
}

int ExceptionFilter(LPEXCEPTION_POINTERS ep )
{
	/*

    int i;
	for( char *ebp = (char *)ep->ContextRecord->Ebp, i = 0; 
			ebp && *((DWORD *)(ebp + 4)) != 0;
			ebp = *(char **)ebp, i++
		)
	{
		int retA = *((int*)(ebp + 4));
		//int relAddr = *((int*)(retA - 4)); 
	}

	*/

	syslog(LOG_ERR, _T("aborted due to win32 exception : '%s' at %x"), 
				ExceptionCodeToString(ep->ExceptionRecord->ExceptionCode), 
				ep->ExceptionRecord->ExceptionAddress );
	//::OutputDebugString(_T("aborted due to win32 exception"));
	
	return EXCEPTION_EXECUTE_HANDLER;
}


unsigned int __stdcall arpd_run_proxy(ArpdConfig* cfg )
{
	__try
	{
		return arpd_run( cfg );
	} __except( ExceptionFilter(GetExceptionInformation()) )
	{
	}
	return 0;
}

unsigned int __stdcall arpd_run( ArpdConfig* cfg )
{
	struct event recv_ev;
	char *dev = 0;

	arpd_init( (TCHAR *)cfg->Devices()[0], cfg->Addresses().Count(), cfg->Addresses().List() );
	arpd_set_respond_on_request_number( cfg->MaxRequestNumber() );

	event_init();
	
	event_set(&recv_ev, pcap_fileno(arpd_pcap), EV_READ,
	    arpd_recv, &recv_ev);
	event_add(&recv_ev, NULL);
	
	/* Setup signal handler */
	if (signal(SIGINT, _terminate_handler) == SIG_ERR) {
		perror("signal");
		return 2;
	}
	if (signal(SIGTERM, _terminate_handler) == SIG_ERR) {
		perror("signal");
		return 2;
	}
	event_sigcb = _signal_handler;
	
	event_dispatch();

	return 0;
}


