/* -/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
 . Copyright (c) 2003 Michael Davis <mike@datanerds.net>
 . All rights reserved.
 .
 . Redistribution and use in source and binary forms, with or without
 . modification, are permitted provided that the following conditions
 . are met:
 .
 . 1. Redistributions of source code must retain the above copyright
 .    notice, this list of conditions and the following disclaimer.
 .
 . 2. Redistributions in binary form must reproduce the above copyright
 .    notice, this list of conditions and the following disclaimer in the
 .    documentation and/or other materials provided with the distribution.
 .
 . 3. The name of author may not be used to endorse or promote products
 .    derived from this software without specific prior written permission.
 .
 . THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 . INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 . AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 . THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 . EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 . PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 . OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 . WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 . OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 . ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 . -\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\-\ */

#include <windows.h>
#include <stdio.h>
#include <time.h>

#include "syslog.h"

#define TBUF_LEN        2048
#define FMT_LEN         1024
#define INTERNALLOG     LOG_ERR|LOG_CONS|LOG_PERROR|LOG_PID

extern "C" int debug = 0;

class CHandle
{
public:
	CHandle( HANDLE h = INVALID_HANDLE_VALUE, BOOL duplicate = FALSE) : handle(INVALID_HANDLE_VALUE)
	{ 
		if (h == INVALID_HANDLE_VALUE)
			return;
		handle = (duplicate) ? Duplicate(h) : h;
	}
	
	CHandle(const CHandle& h) : handle(Duplicate(h.handle))
	{}

	~CHandle()
	{ Close(); }

	operator HANDLE () const
	{ return handle; }
	
	CHandle& operator=(HANDLE  t)
	{ Close(); handle = t; return *this; }

	CHandle& operator=(const CHandle& h)
	{ Close(); handle = Duplicate( h ); return *this; }

	HANDLE Clone(
			DWORD dwOptions = DUPLICATE_SAME_ACCESS,
			DWORD dwDessiredAccess = 0,
			BOOL bInherit = FALSE,
			HANDLE hTragetProcess = ::GetCurrentProcess(), 
			HANDLE hSourceProcess = ::GetCurrentProcess()
		)
	{ return Duplicate(handle,
			dwOptions,
			dwDessiredAccess,
			bInherit,
			hTragetProcess, 
			hSourceProcess
		); 
	}

	BOOL IsOpened() const
	{ return handle != INVALID_HANDLE_VALUE;  }

	void Close()
	{ 
		if (handle != INVALID_HANDLE_VALUE)
			::CloseHandle(handle);
		handle = INVALID_HANDLE_VALUE;
	}

	static HANDLE Duplicate(HANDLE h, 
			DWORD dwOptions = DUPLICATE_SAME_ACCESS,
			DWORD dwDessiredAccess = 0,
			BOOL bInherit = FALSE,
			HANDLE hTragetProcess = ::GetCurrentProcess(), 
			HANDLE hSourceProcess = ::GetCurrentProcess()
			)
	{
		HANDLE hResult = INVALID_HANDLE_VALUE;

		::DuplicateHandle(
			hSourceProcess, 
			h,
			hTragetProcess, 
			&hResult,
			dwDessiredAccess,
			bInherit,
			dwOptions );

		return hResult;
	}

protected:
	HANDLE handle;
};

CHandle logHandle;

void syslog(int pri, const  char *fmt, ...)
{
        va_list ap;
        va_start(ap, fmt);
        vsyslog(pri, fmt, ap);
        va_end(ap);
}



void vsyslog(int pri, const char *fmt, va_list ap)
{
	char ch, *p, *t;
	int tbuf_left, fmt_left, prlen, saved_errno;
	char tbuf[TBUF_LEN], fmt_cpy[FMT_LEN];

	p = tbuf;
	tbuf_left = TBUF_LEN;

	saved_errno = errno;

    /*
     * We wouldn't need this mess if printf handled %m, or if
     * strerror() had been invented before syslog().
     */
    for (t = fmt_cpy, fmt_left = FMT_LEN; (ch = *fmt); ++fmt) 
	{
        if (ch == '%' && fmt[1] == 'm') 
		{
            ++fmt;
            prlen = _snprintf(t, fmt_left, "%s",
                strerror(saved_errno));
            if (prlen >= fmt_left)
                    prlen = fmt_left - 1;
            t += prlen;
            fmt_left -= prlen;
        } 
		else 
		{
            if (fmt_left > 1) 
			{
                *t++ = ch;
                fmt_left--;
            }
        }
    }
    *t = '\0';

	_vsnprintf(p, tbuf_left, fmt_cpy, ap);
	
	/* Get connected, output the message to the local logger. */
	if( 0 != debug )
	{
		printf("%s\n", p);
	}


//#ifdef _DEBUG
		::OutputDebugString(p);
//#endif

	if(logHandle.IsOpened())
	{
		DWORD byteWritten = 0;
		::WriteFile(logHandle, p, strlen(p), &byteWritten, 0);
		::WriteFile(logHandle,"\r\n", 2, &byteWritten, 0);
		
	}

	return;
}

void openlog(const char *fileName, int logstat, int logfac)
{
	logstat; /* not used */
	logfac; /* not used */
	
	logHandle = ::CreateFile(
			fileName,                       // file name
			GENERIC_WRITE,                  // access mode
			FILE_SHARE_READ,				// share mode
			NULL,						    // SD
			OPEN_ALWAYS,					// how to create
			FILE_ATTRIBUTE_NORMAL,          // file attributes
			NULL							// handle to template file
		);

	::SetFilePointer(logHandle, 0, 0, FILE_END);
}

void closelog()
{
	/* there is nothing to do
	 * */

	logHandle.Close();
}
